<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

include 'config.php';
include 'session.php';

// ==========================================
// AJAX HANDLERS
// ==========================================

// Handle AJAX requests
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    switch ($_GET['action']) {
        case 'get_customer':
            try {
                $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
                
                if ($id <= 0) {
                    throw new Exception("ID customer tidak valid");
                }
                
                // Check if new columns exist
                $columns_query = "SHOW COLUMNS FROM customer LIKE 'nama_toko'";
                $columns_result = mysqli_query($conn, $columns_query);
                $has_new_columns = mysqli_num_rows($columns_result) > 0;
                
                $query = "SELECT c.*, 
                                 GROUP_CONCAT(cp.produk_id) as produk_ids,
                                 GROUP_CONCAT(pi.nama SEPARATOR ', ') as produk_names
                          FROM customer c
                          LEFT JOIN customer_produk cp ON cp.customer_id = c.id
                          LEFT JOIN produk_interest pi ON pi.id = cp.produk_id
                          WHERE c.id = ?
                          GROUP BY c.id";
                
                $stmt = mysqli_prepare($conn, $query);
                if (!$stmt) {
                    throw new Exception("Prepare statement failed: " . mysqli_error($conn));
                }
                
                mysqli_stmt_bind_param($stmt, "i", $id);
                
                if (!mysqli_stmt_execute($stmt)) {
                    throw new Exception("Execute failed: " . mysqli_stmt_error($stmt));
                }
                
                $result = mysqli_stmt_get_result($stmt);
                $customer = mysqli_fetch_assoc($result);
                
                if (!$customer) {
                    throw new Exception("Customer dengan ID $id tidak ditemukan");
                }
                
                // Process product IDs
                $customer['produk_id_array'] = [];
                if (!empty($customer['produk_ids'])) {
                    $customer['produk_id_array'] = explode(',', $customer['produk_ids']);
                    $customer['produk_id_array'] = array_map('intval', $customer['produk_id_array']);
                }
                
                // Handle field structure based on database structure
                if ($has_new_columns) {
                    $customer['nama'] = $customer['nama'] ?? '';
                    $customer['nama_toko'] = $customer['nama_toko'] ?? '';
                    $customer['merk_toko'] = $customer['merk_toko'] ?? '';
                } else {
                    $nama_parts = explode(' - ', $customer['nama'] ?? '');
                    $customer['nama'] = $nama_parts[0] ?? '';
                    $customer['nama_toko'] = isset($nama_parts[1]) ? $nama_parts[1] : '';
                    $customer['merk_toko'] = isset($nama_parts[2]) ? $nama_parts[2] : '';
                }
                
                $customer['kota'] = $customer['kota'] ?? '';
                $customer['no_hp'] = $customer['no_hp'] ?? '';
                $customer['jenis'] = $customer['jenis'] ?? '';
                $customer['notes'] = $customer['notes'] ?? '';
                $customer['aktif'] = (int)($customer['aktif'] ?? 1);
                $customer['produk_names'] = $customer['produk_names'] ?? '';
                
                unset($customer['produk_ids']);
                mysqli_stmt_close($stmt);
                
                echo json_encode($customer);
                exit;
                
            } catch (Exception $e) {
                http_response_code(400);
                echo json_encode(['error' => true, 'message' => $e->getMessage()]);
                exit;
            }
            break;
            
        case 'generate_kode':
            try {
                $kota = isset($_GET['kota']) ? trim($_GET['kota']) : '';
                
                if (empty($kota) || strlen($kota) < 3) {
                    echo '';
                    exit;
                }
                
                $kode_prefix = strtoupper(substr($kota, 0, 3));
                
                $query = "SELECT MAX(CAST(SUBSTRING(kode, 4) AS UNSIGNED)) as max_num 
                          FROM customer 
                          WHERE kode LIKE ?";
                
                $stmt = mysqli_prepare($conn, $query);
                if (!$stmt) {
                    echo '';
                    exit;
                }
                
                $like_pattern = $kode_prefix . '%';
                mysqli_stmt_bind_param($stmt, "s", $like_pattern);
                
                if (!mysqli_stmt_execute($stmt)) {
                    echo '';
                    exit;
                }
                
                $result = mysqli_stmt_get_result($stmt);
                $row = mysqli_fetch_assoc($result);
                
                $next_num = ($row['max_num'] ?? 0) + 1;
                $kode = $kode_prefix . str_pad($next_num, 3, '0', STR_PAD_LEFT);
                
                mysqli_stmt_close($stmt);
                echo $kode;
                exit;
                
            } catch (Exception $e) {
                echo '';
                exit;
            }
            break;
            
        case 'add_produk':
            try {
                if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                    throw new Exception("Method tidak diizinkan");
                }
                
                $nama = trim($_POST['nama'] ?? '');
                if (empty($nama)) {
                    throw new Exception("Nama produk tidak boleh kosong");
                }
                
                // Check if product exists
                $check_query = "SELECT id FROM produk_interest WHERE nama = ?";
                $check_stmt = mysqli_prepare($conn, $check_query);
                mysqli_stmt_bind_param($check_stmt, "s", $nama);
                mysqli_stmt_execute($check_stmt);
                $check_result = mysqli_stmt_get_result($check_stmt);
                
                if (mysqli_num_rows($check_result) > 0) {
                    throw new Exception("Produk dengan nama '$nama' sudah ada");
                }
                mysqli_stmt_close($check_stmt);
                
                // Insert new product
                $insert_query = "INSERT INTO produk_interest (nama) VALUES (?)";
                $insert_stmt = mysqli_prepare($conn, $insert_query);
                mysqli_stmt_bind_param($insert_stmt, "s", $nama);
                
                if (!mysqli_stmt_execute($insert_stmt)) {
                    throw new Exception("Gagal menambahkan produk");
                }
                
                $new_id = mysqli_insert_id($conn);
                mysqli_stmt_close($insert_stmt);
                
                echo json_encode([
                    'success' => true,
                    'id' => $new_id,
                    'nama' => $nama
                ]);
                exit;
                
            } catch (Exception $e) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
                exit;
            }
            break;
    }
}

// ==========================================
// SAVE CUSTOMER HANDLER
// ==========================================

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_customer'])) {
    try {
        // Get form data with validation
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $kode = trim($_POST['kode'] ?? '');
        $nama = trim($_POST['nama'] ?? '');
        $nama_toko = trim($_POST['nama_toko'] ?? '');
        $merk_toko = trim($_POST['merk_toko'] ?? '');
        $kota = trim($_POST['kota'] ?? '');
        $no_hp = trim($_POST['no_hp'] ?? '');
        $jenis = trim($_POST['jenis'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
        $aktif = isset($_POST['aktif']) ? (int)$_POST['aktif'] : 1;
        $produk_ids = $_POST['produk_ids'] ?? [];
        $query_string = $_POST['query_string'] ?? '';

        // Validation
        $errors = [];
        
        if (empty($nama)) $errors[] = "Nama person tidak boleh kosong";
        if (empty($kota)) $errors[] = "Kota tidak boleh kosong";
        if (empty($no_hp)) $errors[] = "No. HP tidak boleh kosong";
        if (empty($jenis)) $errors[] = "Jenis customer harus dipilih";
        
        $valid_jenis = ['Sembako', 'Bangunan', 'Sembako dan Bangunan'];
        if (!in_array($jenis, $valid_jenis)) $errors[] = "Jenis customer tidak valid";
        
        if (strlen($nama) > 100) $errors[] = "Nama person maksimal 100 karakter";
        if (strlen($nama_toko) > 100) $errors[] = "Nama toko maksimal 100 karakter";
        if (strlen($merk_toko) > 100) $errors[] = "Merk toko maksimal 100 karakter";
        if (strlen($kota) > 50) $errors[] = "Kota maksimal 50 karakter";
        if (strlen($no_hp) > 20) $errors[] = "No. HP maksimal 20 karakter";
        if (strlen($notes) > 500) $errors[] = "Catatan maksimal 500 karakter";

        if (!empty($errors)) {
            $_SESSION['error_message'] = implode('<br>', $errors);
            $redirect_url = $_SERVER['PHP_SELF'];
            if (!empty($query_string)) {
                $redirect_url .= '?' . $query_string;
            }
            header('Location: ' . $redirect_url);
            exit;
        }

        // Generate kode if not provided
        if (empty($kode) && $id == 0) {
            $kode_prefix = strtoupper(substr($kota, 0, 3));
            $query_kode = "SELECT MAX(CAST(SUBSTRING(kode, 4) AS UNSIGNED)) as max_num 
                           FROM customer 
                           WHERE kode LIKE '" . mysqli_real_escape_string($conn, $kode_prefix) . "%'";
            $result_kode = mysqli_query($conn, $query_kode);
            
            if ($result_kode && $row_kode = mysqli_fetch_assoc($result_kode)) {
                $next_num = ($row_kode['max_num'] ?? 0) + 1;
                $kode = $kode_prefix . str_pad($next_num, 3, '0', STR_PAD_LEFT);
            } else {
                $kode = $kode_prefix . '001';
            }
        }

        // Start transaction
        mysqli_begin_transaction($conn);

        // Check if new columns exist
        $columns_query = "SHOW COLUMNS FROM customer LIKE 'nama_toko'";
        $columns_result = mysqli_query($conn, $columns_query);
        $has_new_columns = mysqli_num_rows($columns_result) > 0;

        if ($id > 0) {
            // Update existing customer
            if ($has_new_columns) {
                $query = "UPDATE customer SET 
                            kode = ?, nama = ?, nama_toko = ?, merk_toko = ?, 
                            kota = ?, no_hp = ?, jenis = ?, notes = ?, aktif = ?
                          WHERE id = ?";
                
                $stmt = mysqli_prepare($conn, $query);
                $nama_toko_value = !empty($nama_toko) ? $nama_toko : null;
                $merk_toko_value = !empty($merk_toko) ? $merk_toko : null;
                $notes_value = !empty($notes) ? $notes : null;
                
                mysqli_stmt_bind_param($stmt, "ssssssssii", 
                    $kode, $nama, $nama_toko_value, $merk_toko_value, 
                    $kota, $no_hp, $jenis, $notes_value, $aktif, $id);
            } else {
                $combined_nama = $nama;
                if (!empty($nama_toko)) $combined_nama .= " - " . $nama_toko;
                if (!empty($merk_toko)) $combined_nama .= " - " . $merk_toko;
                
                $query = "UPDATE customer SET 
                            kode = ?, nama = ?, kota = ?, no_hp = ?, jenis = ?, notes = ?, aktif = ?
                          WHERE id = ?";
                
                $stmt = mysqli_prepare($conn, $query);
                $notes_value = !empty($notes) ? $notes : null;
                
                mysqli_stmt_bind_param($stmt, "ssssssii", 
                    $kode, $combined_nama, $kota, $no_hp, $jenis, $notes_value, $aktif, $id);
            }
            
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            
            // Delete existing products
            $delete_produk = "DELETE FROM customer_produk WHERE customer_id = ?";
            $stmt_delete = mysqli_prepare($conn, $delete_produk);
            mysqli_stmt_bind_param($stmt_delete, "i", $id);
            mysqli_stmt_execute($stmt_delete);
            mysqli_stmt_close($stmt_delete);
            
            $customer_id = $id;
            $action = "diperbarui";
            
        } else {
            // Insert new customer
            if ($has_new_columns) {
                $query = "INSERT INTO customer 
                            (kode, nama, nama_toko, merk_toko, kota, no_hp, jenis, notes, aktif) 
                          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
                
                $stmt = mysqli_prepare($conn, $query);
                $nama_toko_value = !empty($nama_toko) ? $nama_toko : null;
                $merk_toko_value = !empty($merk_toko) ? $merk_toko : null;
                $notes_value = !empty($notes) ? $notes : null;
                
                mysqli_stmt_bind_param($stmt, "ssssssssi", 
                    $kode, $nama, $nama_toko_value, $merk_toko_value, 
                    $kota, $no_hp, $jenis, $notes_value, $aktif);
            } else {
                $combined_nama = $nama;
                if (!empty($nama_toko)) $combined_nama .= " - " . $nama_toko;
                if (!empty($merk_toko)) $combined_nama .= " - " . $merk_toko;
                
                $query = "INSERT INTO customer 
                            (kode, nama, kota, no_hp, jenis, notes, aktif) 
                          VALUES (?, ?, ?, ?, ?, ?, ?)";
                
                $stmt = mysqli_prepare($conn, $query);
                $notes_value = !empty($notes) ? $notes : null;
                
                mysqli_stmt_bind_param($stmt, "ssssssi", 
                    $kode, $combined_nama, $kota, $no_hp, $jenis, $notes_value, $aktif);
            }
            
            mysqli_stmt_execute($stmt);
            $customer_id = mysqli_insert_id($conn);
            mysqli_stmt_close($stmt);
            
            $action = "ditambahkan";
        }

        // Insert product relationships
        if (!empty($produk_ids) && is_array($produk_ids)) {
            $table_check = "SHOW TABLES LIKE 'customer_produk'";
            $table_result = mysqli_query($conn, $table_check);
            
            if (mysqli_num_rows($table_result) > 0) {
                $insert_produk = "INSERT INTO customer_produk (customer_id, produk_id) VALUES (?, ?)";
                $stmt_produk = mysqli_prepare($conn, $insert_produk);
                
                foreach ($produk_ids as $produk_id) {
                    $produk_id = (int)$produk_id;
                    if ($produk_id > 0) {
                        mysqli_stmt_bind_param($stmt_produk, "ii", $customer_id, $produk_id);
                        mysqli_stmt_execute($stmt_produk);
                    }
                }
                mysqli_stmt_close($stmt_produk);
            }
        }

        // Commit transaction
        mysqli_commit($conn);
        
        // Success message
        $display_name = $nama;
        if (!empty($nama_toko)) {
            $display_name .= " (" . $nama_toko . ")";
        }
        
        $_SESSION['success_message'] = "Customer '$display_name' berhasil $action dengan kode: $kode";
        
        // Redirect
        $redirect_url = $_SERVER['PHP_SELF'];
        if (!empty($query_string)) {
            $redirect_url .= '?' . $query_string;
        }
        header('Location: ' . $redirect_url);
        exit;

    } catch (Exception $e) {
        mysqli_rollback($conn);
        $_SESSION['error_message'] = "Error menyimpan data customer: " . $e->getMessage();
        
        $redirect_url = $_SERVER['PHP_SELF'];
        if (!empty($_POST['query_string'])) {
            $redirect_url .= '?' . $_POST['query_string'];
        }
        header('Location: ' . $redirect_url);
        exit;
    }
}

// ==========================================
// MAIN PAGE LOGIC
// ==========================================

// Get filter parameters
$filter_nama = isset($_GET['nama']) ? trim($_GET['nama']) : '';
$filter_nama_toko = isset($_GET['nama_toko']) ? trim($_GET['nama_toko']) : '';
$filter_merk_toko = isset($_GET['merk_toko']) ? trim($_GET['merk_toko']) : '';
$filter_kota = isset($_GET['kota']) ? trim($_GET['kota']) : '';
$filter_produk = isset($_GET['produk']) ? trim($_GET['produk']) : '';
$filter_notes = isset($_GET['notes']) ? trim($_GET['notes']) : '';
$filter_jenis = isset($_GET['jenis']) ? trim($_GET['jenis']) : '';
$filter_aktif = isset($_GET['aktif']) ? $_GET['aktif'] : '';

// Build WHERE clause
$where = [];
if ($filter_nama !== '') $where[] = "c.nama LIKE '%" . mysqli_real_escape_string($conn, $filter_nama) . "%'";
if ($filter_nama_toko !== '') $where[] = "c.nama_toko LIKE '%" . mysqli_real_escape_string($conn, $filter_nama_toko) . "%'";
if ($filter_merk_toko !== '') $where[] = "c.merk_toko LIKE '%" . mysqli_real_escape_string($conn, $filter_merk_toko) . "%'";
if ($filter_kota !== '') $where[] = "c.kota LIKE '%" . mysqli_real_escape_string($conn, $filter_kota) . "%'";
if ($filter_produk !== '') $where[] = "pi.nama LIKE '%" . mysqli_real_escape_string($conn, $filter_produk) . "%'";
if ($filter_notes !== '') $where[] = "c.notes LIKE '%" . mysqli_real_escape_string($conn, $filter_notes) . "%'";
if ($filter_jenis !== '') $where[] = "c.jenis = '" . mysqli_real_escape_string($conn, $filter_jenis) . "'";
if ($filter_aktif !== '') $where[] = "c.aktif = " . (int)$filter_aktif;

$where_sql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

// Get customer data
try {
    $query = "SELECT c.*, GROUP_CONCAT(pi.nama SEPARATOR ', ') AS produk_interest
              FROM customer c
              LEFT JOIN customer_produk cp ON cp.customer_id = c.id
              LEFT JOIN produk_interest pi ON pi.id = cp.produk_id
              $where_sql
              GROUP BY c.id
              ORDER BY c.id ASC";
    
    $result = mysqli_query($conn, $query);
    if (!$result) {
        throw new Exception("Error fetching customer data: " . mysqli_error($conn));
    }
    $totalCustomer = mysqli_num_rows($result);
} catch (Exception $e) {
    $_SESSION['error_message'] = "Error mengambil data customer: " . $e->getMessage();
    $totalCustomer = 0;
    $result = false;
}

// Get product list
try {
    $produk_list = mysqli_query($conn, "SELECT * FROM produk_interest ORDER BY nama ASC");
} catch (Exception $e) {
    $produk_list = false;
}

// Check if new columns exist
$columns_query = "SHOW COLUMNS FROM customer LIKE 'nama_toko'";
$columns_result = mysqli_query($conn, $columns_query);
$has_new_columns = mysqli_num_rows($columns_result) > 0;
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Master Customer</title>
    <link rel="icon" type="image/x-icon" href="/images/favicon.ico">
    <style>
        .produk-tag {
            display: inline-block;
            padding: 4px 10px;
            background-color: #e3f2fd;
            color: #1976d2;
            border: 1px solid #bbdefb;
            border-radius: 20px;
            margin: 2px;
            font-size: 12px;
            font-weight: 500;
        }
        .produk-tag .remove {
            margin-left: 6px;
            color: #dc3545;
            cursor: pointer;
            font-weight: bold;
        }
        .produk-tag .remove:hover {
            color: #c82333;
        }
        .filter-card {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 20px;
        }
        .badge-jenis {
            font-size: 11px;
            padding: 4px 8px;
        }
        .table td {
            vertical-align: middle;
        }
        .btn-action {
            padding: 4px 8px;
            border: none;
            margin: 0 2px;
            cursor: pointer;
            border-radius: 3px;
            display: inline-block;
            font-size: 11px;
            color: white;
            text-decoration: none;
        }
        .btn-edit {
            background-color: #ffc107;
            color: #000;
        }
        .btn-edit:hover {
            background-color: #e0a800;
            color: #000;
        }
        .customer-info {
            line-height: 1.3;
        }
        .customer-name {
            font-weight: bold;
            color: #2c3e50;
        }
        .shop-name {
            color: #27ae60;
            font-size: 0.9em;
        }
        .brand-name {
            color: #e67e22;
            font-size: 0.85em;
            font-style: italic;
        }
    </style>
</head>
<body>

<!-- Include Menu Navigation -->
<?php 
if (!isset($jabatan_relasi)) {
    $jabatan_relasi = $_SESSION['jabatan'] ?? '';
}
include 'navbar.php'; 
?>

<div class="container-fluid py-4">
    
    <!-- Page Header -->
    <div class="row mb-4">
        <div class="col">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h3 class="mb-1 text-dark">
                        <i class="fa fa-users text-primary"></i> Master Customer
                        <?php if (!$has_new_columns): ?>
                            <span class="badge bg-warning text-dark">Legacy Mode</span>
                        <?php endif; ?>
                    </h3>
                    <p class="text-muted mb-0">Kelola data customer perusahaan</p>
                </div>
                <div>
                    <span class="badge bg-primary fs-6">
                        <i class="fa fa-database"></i> <?= $totalCustomer ?> Customer
                    </span>
                </div>
            </div>
        </div>
    </div>

    <!-- Alert Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fa fa-check-circle"></i> <?= htmlspecialchars($_SESSION['success_message']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fa fa-exclamation-circle"></i> <?= htmlspecialchars($_SESSION['error_message']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>

    <?php if (!$has_new_columns): ?>
        <div class="alert alert-info alert-dismissible fade show" role="alert">
            <i class="fa fa-info-circle"></i> 
            <strong>Info:</strong> Database masih menggunakan struktur lama. 
            Jalankan migration untuk memisahkan field nama, nama toko, dan merk toko.
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Filter Section -->
    <div class="card shadow-sm mb-4">
        <div class="card-header bg-light">
            <h6 class="mb-0">
                <i class="fa fa-filter text-secondary"></i> Filter & Pencarian Data
            </h6>
        </div>
        <div class="card-body">
            <form method="get" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label small">Nama Person</label>
                        <input type="text" name="nama" class="form-control" 
                               placeholder="Cari nama person..." 
                               value="<?= htmlspecialchars($filter_nama) ?>">
                    </div>
                    <?php if ($has_new_columns): ?>
                    <div class="col-md-3">
                        <label class="form-label small">Nama Toko</label>
                        <input type="text" name="nama_toko" class="form-control" 
                               placeholder="Cari nama toko..." 
                               value="<?= htmlspecialchars($filter_nama_toko) ?>">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label small">Merk Toko</label>
                        <input type="text" name="merk_toko" class="form-control" 
                               placeholder="Cari merk..." 
                               value="<?= htmlspecialchars($filter_merk_toko) ?>">
                    </div>
                    <?php endif; ?>
                    <div class="col-md-2">
                        <label class="form-label small">Kota</label>
                        <input type="text" name="kota" class="form-control" 
                               placeholder="Cari kota..." 
                               value="<?= htmlspecialchars($filter_kota) ?>">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label small">Produk Interest</label>
                        <input type="text" name="produk" class="form-control" 
                               placeholder="Cari produk..." 
                               value="<?= htmlspecialchars($filter_produk) ?>">
                    </div>

                    <div class="col-md-3">
                        <label class="form-label small">Jenis</label>
                        <select name="jenis" class="form-select">
                            <option value="">Semua Jenis</option>
                            <option value="Sembako" <?= $filter_jenis === 'Sembako' ? 'selected' : '' ?>>Sembako</option>
                            <option value="Bangunan" <?= $filter_jenis === 'Bangunan' ? 'selected' : '' ?>>Bangunan</option>
                            <option value="Sembako dan Bangunan" <?= $filter_jenis === 'Sembako dan Bangunan' ? 'selected' : '' ?>>Sembako dan Bangunan</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label small">Status</label>
                        <select name="aktif" class="form-select">
                            <option value="">Semua Status</option>
                            <option value="1" <?= $filter_aktif === '1' ? 'selected' : '' ?>>Aktif</option>
                            <option value="0" <?= $filter_aktif === '0' ? 'selected' : '' ?>>Tidak Aktif</option>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label small">Catatan</label>
                        <input type="text" name="notes" class="form-control" 
                               placeholder="Cari dalam catatan..." 
                               value="<?= htmlspecialchars($filter_notes) ?>">
                    </div>

                    <div class="col-12">
                        <button type="submit" class="btn btn-secondary me-2">
                            <i class="fa fa-filter"></i> Terapkan Filter
                        </button>
                        <a href="<?= $_SERVER['PHP_SELF'] ?>" class="btn btn-outline-secondary">
                            <i class="fa fa-refresh"></i> Reset Filter
                        </a>
                    </div>
                </form>
        </div>
    </div>

    <!-- Action Buttons -->
    <div class="row mb-3">
        <div class="col-md-6">
            <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#customerModal" onclick="openAddModal()">
                <i class="fa fa-plus"></i> Tambah Customer Baru
            </button>
        </div>
        <div class="col-md-6 text-end">
            <?php if ($filter_nama || $filter_nama_toko || $filter_merk_toko || $filter_kota || $filter_produk || $filter_notes || $filter_jenis || $filter_aktif !== ''): ?>
                <span class="badge bg-info">
                    <i class="fa fa-filter"></i> Filter Aktif
                </span>
            <?php endif; ?>
        </div>
    </div>

    <!-- Data Table -->
    <div class="row">
        <div class="col">
            <div class="card shadow-sm">
                <div class="card-header bg-dark text-white">
                    <h5 class="mb-0">
                        <i class="fa fa-list"></i> Daftar Customer
                    </h5>
                </div>
                <div class="card-body p-0">
                    <?php if ($result && $totalCustomer > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th width="50" class="text-center">#</th>
                                        <th width="80">Kode</th>
                                        <th width="200">Informasi Customer</th>
                                        <th width="100">Kota</th>
                                        <th width="120">No. HP</th>
                                        <th width="150">Catatan</th>
                                        <th width="120">Jenis</th>
                                        <th width="80" class="text-center">Status</th>
                                        <th width="150" class="text-center">Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $no = 1;
                                    mysqli_data_seek($result, 0);
                                    while ($row = mysqli_fetch_assoc($result)): 
                                        // Handle different database structures
                                        if ($has_new_columns) {
                                            $nama_person = $row['nama'] ?? '';
                                            $nama_toko = $row['nama_toko'] ?? '';
                                            $merk_toko = $row['merk_toko'] ?? '';
                                        } else {
                                            // Try to parse combined name
                                            $nama_parts = explode(' - ', $row['nama'] ?? '');
                                            $nama_person = $nama_parts[0] ?? '';
                                            $nama_toko = isset($nama_parts[1]) ? $nama_parts[1] : '';
                                            $merk_toko = isset($nama_parts[2]) ? $nama_parts[2] : '';
                                        }
                                    ?>
                                        <tr>
                                            <td class="text-center">
                                                <span class="badge bg-secondary"><?= $no++ ?></span>
                                            </td>
                                            <td>
                                                <code class="text-primary"><?= htmlspecialchars($row['kode']) ?></code>
                                            </td>
                                            <td>
                                                <div class="customer-info">
                                                    <div class="customer-name"><?= htmlspecialchars($nama_person ?: '-') ?></div>
                                                    <?php if (!empty($nama_toko)): ?>
                                                        <div class="shop-name">
                                                            <i class="fa fa-store fa-xs"></i> <?= htmlspecialchars($nama_toko) ?>
                                                        </div>
                                                    <?php endif; ?>
                                                    <?php if (!empty($merk_toko)): ?>
                                                        <div class="brand-name">
                                                            <i class="fa fa-tag fa-xs"></i> <?= htmlspecialchars($merk_toko) ?>
                                                        </div>
                                                    <?php endif; ?>
                                                    <?php if (!empty($row['produk_interest'])): ?>
                                                        <div class="mt-1">
                                                            <small class="text-muted">
                                                                <i class="fa fa-boxes fa-xs"></i> 
                                                                <?= htmlspecialchars(substr($row['produk_interest'], 0, 30)) ?>
                                                                <?= strlen($row['produk_interest']) > 30 ? '...' : '' ?>
                                                            </small>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="text-muted"><?= htmlspecialchars($row['kota']) ?></span>
                                            </td>
                                            <td>
                                                <span class="text-muted"><?= htmlspecialchars($row['no_hp']) ?></span>
                                            </td>
                                            <td>
                                                <span class="text-muted small">
                                                    <?= !empty($row['notes']) ? htmlspecialchars(substr($row['notes'], 0, 30) . (strlen($row['notes']) > 30 ? '...' : '')) : '<em class="text-secondary">-</em>' ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if (!empty($row['jenis'])): ?>
                                                    <?php
                                                    $badge_class = '';
                                                    switch($row['jenis']) {
                                                        case 'Sembako': $badge_class = 'bg-success'; break;
                                                        case 'Bangunan': $badge_class = 'bg-warning text-dark'; break;
                                                        case 'Sembako dan Bangunan': $badge_class = 'bg-info'; break;
                                                        default: $badge_class = 'bg-secondary';
                                                    }
                                                    ?>
                                                    <span class="badge <?= $badge_class ?> badge-jenis">
                                                        <?= htmlspecialchars($row['jenis']) ?>
                                                    </span>
                                                <?php else: ?>
                                                    <em class="text-muted">-</em>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-center">
                                                <?php if ($row['aktif']): ?>
                                                    <span class="badge bg-success">
                                                        <i class="fa fa-check"></i> Aktif
                                                    </span>
                                                <?php else: ?>
                                                    <span class="badge bg-danger">
                                                        <i class="fa fa-times"></i> Nonaktif
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-center">
                                                <button class="btn-action btn-edit" 
                                                        onclick="editCustomer(<?= (int)$row['id'] ?>, '<?= htmlspecialchars($_SERVER['QUERY_STRING'], ENT_QUOTES) ?>')"
                                                        title="Edit Customer">
                                                    Edit
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-5">
                            <i class="fa fa-search fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">
                                <?php if ($result === false): ?>
                                    Error mengambil data
                                <?php elseif ($filter_nama || $filter_nama_toko || $filter_merk_toko || $filter_kota || $filter_produk || $filter_notes || $filter_jenis || $filter_aktif !== ''): ?>
                                    Tidak ada data sesuai filter
                                <?php else: ?>
                                    Belum ada data customer
                                <?php endif; ?>
                            </h5>
                            <p class="text-muted">
                                <?php if ($result === false): ?>
                                    Silakan refresh halaman atau hubungi administrator
                                <?php elseif ($filter_nama || $filter_nama_toko || $filter_merk_toko || $filter_kota || $filter_produk || $filter_notes || $filter_jenis || $filter_aktif !== ''): ?>
                                    Coba ubah kriteria pencarian atau reset filter
                                <?php else: ?>
                                    Klik tombol "Tambah Customer Baru" untuk menambah data pertama
                                <?php endif; ?>
                            </p>
                            <?php if ($result !== false && !($filter_nama || $filter_nama_toko || $filter_merk_toko || $filter_kota || $filter_produk || $filter_notes || $filter_jenis || $filter_aktif !== '')): ?>
                                <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#customerModal" onclick="openAddModal()">
                                    <i class="fa fa-plus"></i> Tambah Sekarang
                                </button>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
                <?php if ($result && $totalCustomer > 0): ?>
                    <div class="card-footer bg-light">
                        <small class="text-muted">
                            <i class="fa fa-info-circle"></i> 
                            Menampilkan <strong><?= $totalCustomer ?></strong> data customer
                            <?php if ($filter_nama || $filter_nama_toko || $filter_merk_toko || $filter_kota || $filter_produk || $filter_notes || $filter_jenis || $filter_aktif !== ''): ?>
                                dari hasil filter
                            <?php endif; ?>
                        </small>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

</div>

<!-- Modal Add/Edit Customer -->
<div class="modal fade" id="customerModal" tabindex="-1" aria-labelledby="modalTitle" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <form method="post" action="" id="customerForm" onsubmit="return validateForm()">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="modalTitle">
                        <i class="fa fa-plus"></i> Tambah Customer
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="save_customer" value="1">
                    <input type="hidden" name="id" id="customerId">
                    <input type="hidden" name="query_string" id="query_string">

                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="kode" class="form-label">
                                    <i class="fa fa-barcode"></i> Kode Customer
                                </label>
                                <input type="text" class="form-control" name="kode" id="kode" readonly>
                                <div class="form-text">Kode otomatis berdasarkan kota</div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="nama" class="form-label">
                                    <i class="fa fa-user"></i> Nama Person <span class="text-danger">*</span>
                                </label>
                                <input type="text" class="form-control" name="nama" id="nama" 
                                       placeholder="Nama orang/pemilik..." 
                                       required maxlength="100" autocomplete="off">
                                <div class="invalid-feedback" id="namaError"></div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="kota" class="form-label">
                                    <i class="fa fa-map-marker-alt"></i> Kota <span class="text-danger">*</span>
                                </label>
                                <input type="text" class="form-control" name="kota" id="kota" 
                                       placeholder="Masukkan kota..." 
                                       required maxlength="50" autocomplete="off">
                                <div class="form-text">Min. 3 karakter untuk generate kode</div>
                                <div class="invalid-feedback" id="kotaError"></div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nama_toko" class="form-label">
                                    <i class="fa fa-store"></i> Nama Toko
                                </label>
                                <input type="text" class="form-control" name="nama_toko" id="nama_toko" 
                                       placeholder="Nama toko/usaha..." 
                                       maxlength="100" autocomplete="off">
                                <div class="form-text">Opsional - nama toko/usaha customer</div>
                                <div class="invalid-feedback" id="namaTokoError"></div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="merk_toko" class="form-label">
                                    <i class="fa fa-tag"></i> Merk Toko
                                </label>
                                <input type="text" class="form-control" name="merk_toko" id="merk_toko" 
                                       placeholder="Brand/merk toko..." 
                                       maxlength="100" autocomplete="off">
                                <div class="form-text">Opsional - merk/brand toko</div>
                                <div class="invalid-feedback" id="merkTokoError"></div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="no_hp" class="form-label">
                                    <i class="fa fa-phone"></i> No. HP <span class="text-danger">*</span>
                                </label>
                                <input type="text" class="form-control" name="no_hp" id="no_hp" 
                                       placeholder="Contoh: 08123456789" 
                                       required maxlength="20" autocomplete="off">
                                <div class="invalid-feedback" id="noHpError"></div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="jenis" class="form-label">
                                    <i class="fa fa-tags"></i> Jenis <span class="text-danger">*</span>
                                </label>
                                <select class="form-select" name="jenis" id="jenis" required>
                                    <option value="">-- Pilih Jenis --</option>
                                    <option value="Sembako">Sembako</option>
                                    <option value="Bangunan">Bangunan</option>
                                    <option value="Sembako dan Bangunan">Sembako dan Bangunan</option>
                                </select>
                                <div class="invalid-feedback" id="jenisError"></div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">
                                    <i class="fa fa-toggle-on"></i> Status
                                </label>
                                <div class="mt-2">
                                    <div class="form-check form-check-inline">
                                        <input class="form-check-input" type="radio" name="aktif" id="aktif1" value="1" checked>
                                        <label class="form-check-label" for="aktif1">
                                            <span class="badge bg-success">Aktif</span>
                                        </label>
                                    </div>
                                    <div class="form-check form-check-inline">
                                        <input class="form-check-input" type="radio" name="aktif" id="aktif0" value="0">
                                        <label class="form-check-label" for="aktif0">
                                            <span class="badge bg-danger">Tidak Aktif</span>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="notes" class="form-label">
                            <i class="fa fa-sticky-note"></i> Catatan / Notes
                        </label>
                        <textarea class="form-control" name="notes" id="notes" rows="3" 
                                  placeholder="Masukkan catatan tentang customer..." 
                                  maxlength="500"></textarea>
                        <div class="form-text">Maksimal 500 karakter</div>
                        <div class="invalid-feedback" id="notesError"></div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">
                            <i class="fa fa-box"></i> Produk Interest
                        </label>
                        <div id="produk_container" class="mb-2 border rounded p-2 bg-light" style="min-height: 50px;">
                            <small class="text-muted">Produk yang dipilih akan muncul di sini...</small>
                        </div>
                        <div class="input-group">
                            <select class="form-select" id="produk_select">
                                <option value="">-- Pilih Produk Interest --</option>
                                <?php if ($produk_list): ?>
                                    <?php mysqli_data_seek($produk_list, 0); while ($produk = mysqli_fetch_assoc($produk_list)): ?>
                                        <option value="<?= (int)$produk['id'] ?>"><?= htmlspecialchars($produk['nama']) ?></option>
                                    <?php endwhile; ?>
                                <?php endif; ?>
                            </select>
                            <button type="button" class="btn btn-outline-success" onclick="promptProdukBaru()" title="Tambah produk baru">
                                <i class="fa fa-plus"></i>
                            </button>
                        </div>
                        <div class="form-text">Pilih produk dari dropdown atau tambah produk baru</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="submit" class="btn btn-success" id="submitBtn">
                        <i class="fa fa-save"></i> Simpan Customer
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fa fa-times"></i> Batal
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<script>
let selectedProduk = [];

// Validation function
function validateForm() {
    const nama = document.getElementById('nama');
    const kota = document.getElementById('kota');
    const noHp = document.getElementById('no_hp');
    const jenis = document.getElementById('jenis');
    const submitBtn = document.getElementById('submitBtn');
    
    // Reset validation
    [nama, kota, noHp, jenis].forEach(field => {
        field.classList.remove('is-invalid');
        const errorElement = document.getElementById(field.id + 'Error');
        if (errorElement) errorElement.textContent = '';
    });
    
    let isValid = true;
    
    // Validate nama
    if (nama.value.trim() === '') {
        nama.classList.add('is-invalid');
        document.getElementById('namaError').textContent = 'Nama person tidak boleh kosong!';
        if (isValid) nama.focus();
        isValid = false;
    }
    
    // Validate kota
    if (kota.value.trim() === '') {
        kota.classList.add('is-invalid');
        document.getElementById('kotaError').textContent = 'Kota tidak boleh kosong!';
        if (isValid) kota.focus();
        isValid = false;
    }
    
    // Validate no HP
    if (noHp.value.trim() === '') {
        noHp.classList.add('is-invalid');
        document.getElementById('noHpError').textContent = 'No. HP tidak boleh kosong!';
        if (isValid) noHp.focus();
        isValid = false;
    }
    
    // Validate jenis
    if (jenis.value === '') {
        jenis.classList.add('is-invalid');
        document.getElementById('jenisError').textContent = 'Jenis customer harus dipilih!';
        if (isValid) jenis.focus();
        isValid = false;
    }
    
    if (isValid) {
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Menyimpan...';
    }
    
    return isValid;
}

// Render product tags
function renderProdukTags() {
    const container = document.getElementById('produk_container');
    
    if (selectedProduk.length === 0) {
        container.innerHTML = '<small class="text-muted">Produk yang dipilih akan muncul di sini...</small>';
    } else {
        container.innerHTML = '';
        selectedProduk.forEach(p => {
            const tag = document.createElement('span');
            tag.className = 'produk-tag';
            tag.innerHTML = `${p.nama} <span class="remove" onclick="removeProduk(${p.id})" title="Hapus">&times;</span>`;
            container.appendChild(tag);
        });
    }

    // Remove existing hidden inputs
    document.querySelectorAll('input[name="produk_ids[]"]').forEach(el => el.remove());

    // Add hidden inputs for selected products
    selectedProduk.forEach(p => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'produk_ids[]';
        input.value = p.id;
        document.getElementById('customerForm').appendChild(input);
    });
}

// Remove product from selection
function removeProduk(id) {
    selectedProduk = selectedProduk.filter(p => p.id != id);
    renderProdukTags();
}

// Add product from dropdown
document.getElementById('produk_select').addEventListener('change', function() {
    const id = this.value;
    const nama = this.options[this.selectedIndex].text;
    if (!id || selectedProduk.find(p => p.id == id)) {
        this.value = '';
        return;
    }
    selectedProduk.push({ id, nama });
    renderProdukTags();
    this.value = '';
});

// Open add modal
function openAddModal() {
    document.getElementById('modalTitle').innerHTML = '<i class="fa fa-plus"></i> Tambah Customer';
    document.getElementById('customerId').value = '';
    document.getElementById('kode').value = '';
    document.getElementById('nama').value = '';
    document.getElementById('nama_toko').value = '';
    document.getElementById('merk_toko').value = '';
    document.getElementById('kota').value = '';
    document.getElementById('no_hp').value = '';
    document.getElementById('notes').value = '';
    document.getElementById('jenis').value = '';
    document.getElementById('aktif1').checked = true;
    
    // Reset validation
    ['nama', 'nama_toko', 'merk_toko', 'kota', 'no_hp', 'jenis'].forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) field.classList.remove('is-invalid');
    });
    
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fa fa-save"></i> Simpan Customer';
    }

    selectedProduk = [];
    renderProdukTags();
}

// Generate customer code based on city
function fetchKodeCustomer(kota) {
    const trimmed = kota.trim();
    if (trimmed.length < 3) {
        document.getElementById('kode').value = '';
        return;
    }

    fetch('?action=generate_kode&kota=' + encodeURIComponent(trimmed))
        .then(res => res.text())
        .then(kode => {
            document.getElementById('kode').value = kode;
        })
        .catch(err => {
            console.error('Error generating customer code:', err);
        });
}

// City input listener
document.getElementById('kota').addEventListener('input', function () {
    fetchKodeCustomer(this.value);
});

// Edit customer
function editCustomer(id, queryString) {
    fetch('?action=get_customer&id=' + id)
        .then(res => res.json())
        .then(data => {
            if (data.error) {
                throw new Error(data.message);
            }
            
            document.getElementById('modalTitle').innerHTML = '<i class="fa fa-edit"></i> Edit Customer';
            document.getElementById('customerId').value = data.id;
            document.getElementById('kode').value = data.kode;
            document.getElementById('nama').value = data.nama || '';
            document.getElementById('nama_toko').value = data.nama_toko || '';
            document.getElementById('merk_toko').value = data.merk_toko || '';
            document.getElementById('kota').value = data.kota;
            document.getElementById('jenis').value = data.jenis || '';
            document.getElementById('no_hp').value = data.no_hp;
            document.getElementById('notes').value = data.notes || '';
            document.getElementById(data.aktif == 1 ? 'aktif1' : 'aktif0').checked = true;

            // Reset validation
            ['nama', 'nama_toko', 'merk_toko', 'kota', 'no_hp', 'jenis'].forEach(fieldId => {
                const field = document.getElementById(fieldId);
                if (field) field.classList.remove('is-invalid');
            });
            
            const submitBtn = document.getElementById('submitBtn');
            if (submitBtn) {
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fa fa-save"></i> Update Customer';
            }

            selectedProduk = [];
            renderProdukTags();

            // Set query string
            document.getElementById('query_string').value = queryString;

            let modal = new bootstrap.Modal(document.getElementById('customerModal'));
            modal.show();
        })
        .catch(err => {
            console.error('Error fetching customer data:', err);
            alert('Error mengambil data customer: ' + err.message);
        });
}

// Add new product
function promptProdukBaru() {
    let nama = prompt("Nama Produk Interest Baru:");
    if (!nama || nama.trim() === '') return;

    nama = nama.trim();

    // Check if product already exists in dropdown
    let existing = Array.from(document.querySelectorAll('#produk_select option'))
        .find(opt => opt.text.toLowerCase() === nama.toLowerCase());
    if (existing) {
        alert("Produk dengan nama '" + nama + "' sudah ada.");
        return;
    }

    fetch('?action=add_produk', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ nama })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success && data.id) {
            // Add to dropdown
            let option = new Option(data.nama, data.id);
            document.getElementById('produk_select').append(option);
            
            // Add to selected products
            selectedProduk.push({ id: data.id, nama: data.nama });
            renderProdukTags();
            
            // Show success message
            const alertHtml = `
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fa fa-check-circle"></i> Produk "${data.nama}" berhasil ditambahkan!
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            `;
            document.querySelector('.modal-body').insertAdjacentHTML('afterbegin', alertHtml);
            
            // Auto-hide alert after 3 seconds
            setTimeout(() => {
                const alert = document.querySelector('.modal-body .alert');
                if (alert) alert.remove();
            }, 3000);
        } else {
            alert("Gagal menambahkan produk baru: " + (data.message || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Error adding new product:', err);
        alert("Gagal menambahkan produk baru");
    });
}

// Auto-hide alerts after 5 seconds
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert-dismissible');
    alerts.forEach(function(alert) {
        setTimeout(function() {
            if (alert && alert.parentNode) {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            }
        }, 5000);
    });
    
    // Real-time validation
    const inputs = ['nama', 'nama_toko', 'merk_toko', 'kota', 'no_hp', 'jenis', 'notes'];
    inputs.forEach(function(inputId) {
        const input = document.getElementById(inputId);
        if (input) {
            input.addEventListener('input', function() {
                if (this.classList.contains('is-invalid')) {
                    this.classList.remove('is-invalid');
                    const errorElement = document.getElementById(inputId + 'Error');
                    if (errorElement) errorElement.textContent = '';
                }
            });
        }
    });
    
    // Initialize product tags
    renderProdukTags();
});

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl+Enter untuk submit form
    if (e.key === 'Enter' && e.ctrlKey) {
        const activeModal = document.querySelector('.modal.show');
        if (activeModal) {
            const submitBtn = activeModal.querySelector('button[type="submit"]');
            if (submitBtn && !submitBtn.disabled) {
                submitBtn.click();
            }
        }
    }
    
    // Ctrl+N untuk tambah baru
    if (e.key === 'n' && e.ctrlKey && !e.shiftKey && !e.altKey) {
        e.preventDefault();
        const addBtn = document.querySelector('[data-bs-target="#customerModal"]');
        if (addBtn) addBtn.click();
    }
    
    // F5 untuk refresh (override default)
    if (e.key === 'F5') {
        e.preventDefault();
        location.reload();
    }
});

// Modal event listeners
document.getElementById('customerModal').addEventListener('shown.bs.modal', function () {
    const namaInput = document.getElementById('nama');
    if (namaInput) namaInput.focus();
});

// Reset form when modal is hidden
document.getElementById('customerModal').addEventListener('hidden.bs.modal', function () {
    const form = document.getElementById('customerForm');
    if (form) {
        // Reset form validation
        ['nama', 'nama_toko', 'merk_toko', 'kota', 'no_hp', 'jenis'].forEach(fieldId => {
            const field = document.getElementById(fieldId);
            if (field) field.classList.remove('is-invalid');
        });
        
        // Reset submit button
        const submitBtn = document.getElementById('submitBtn');
        if (submitBtn) {
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fa fa-save"></i> Simpan Customer';
        }
    }
});
</script>

</body>
</html>