<?php
// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include files
include 'config.php';
include 'session.php';
include 'navbar.php';

// Check database connection
if (!isset($conn) || !$conn) {
    die("Error: Koneksi database tidak tersedia");
}

// Parameters
$periode = isset($_GET['periode']) ? $_GET['periode'] : 'all';
$customer_id = isset($_GET['customer_id']) ? (int)$_GET['customer_id'] : 0;
$kota = isset($_GET['kota']) ? $_GET['kota'] : '';
$export = isset($_GET['export']) ? $_GET['export'] : '';

// Functions
function formatCurrency($amount) {
    return 'Rp ' . number_format($amount, 0, ',', '.');
}

function formatTanggal($tanggal) {
    return date('d/m/Y', strtotime($tanggal));
}

// Build query conditions
$where_conditions = ["c.aktif = 1"];
if ($periode != 'all') {
    $where_conditions[] = "DATE_FORMAT(p.tanggal, '%Y-%m') = '$periode'";
}
if ($customer_id > 0) {
    $where_conditions[] = "c.id = $customer_id";
}
if (!empty($kota)) {
    $where_conditions[] = "c.kota = '" . mysqli_real_escape_string($conn, $kota) . "'";
}
$where_clause = "WHERE " . implode(" AND ", $where_conditions);

// Main query - Customer ranking
$ranking_query = "
    SELECT 
        c.id,
        c.kode as customer_kode,
        c.nama as customer_nama,
        c.nama_toko,
        c.kota,
        c.jenis,
        COUNT(DISTINCT p.id) as total_transaksi,
        SUM(pd.qty * pd.harga) as total_nominal,
        AVG(pd.qty * pd.harga) as rata_rata_transaksi,
        MAX(p.tanggal) as transaksi_terakhir,
        MIN(p.tanggal) as transaksi_pertama,
        SUM((pd.qty * pd.harga) - (pd.qty * COALESCE(pd.harga_beli, 0))) as total_laba,
        CASE 
            WHEN SUM(pd.qty * pd.harga) > 0 THEN 
                (SUM((pd.qty * pd.harga) - (pd.qty * COALESCE(pd.harga_beli, 0))) / SUM(pd.qty * pd.harga) * 100)
            ELSE 0 
        END as margin_persen
    FROM customer c
    LEFT JOIN penjualan p ON c.id = p.customer_id
    LEFT JOIN penjualan_detail pd ON p.id = pd.penjualan_id
    $where_clause AND p.id IS NOT NULL
    GROUP BY c.id, c.kode, c.nama, c.nama_toko, c.kota, c.jenis
    HAVING total_transaksi > 0
    ORDER BY total_nominal DESC
    LIMIT 50
";

$ranking_result = mysqli_query($conn, $ranking_query);
if (!$ranking_result) {
    die("Ranking query error: " . mysqli_error($conn));
}

// Summary query
$summary_query = "
    SELECT 
        COUNT(DISTINCT c.id) as total_customer_aktif,
        COUNT(DISTINCT p.id) as total_transaksi,
        SUM(pd.qty * pd.harga) as grand_total_penjualan,
        SUM((pd.qty * pd.harga) - (pd.qty * COALESCE(pd.harga_beli, 0))) as total_laba,
        AVG(CASE 
            WHEN (pd.qty * pd.harga) > 0 THEN 
                (((pd.qty * pd.harga) - (pd.qty * COALESCE(pd.harga_beli, 0))) / (pd.qty * pd.harga) * 100)
            ELSE 0 
        END) as rata_rata_margin
    FROM customer c
    LEFT JOIN penjualan p ON c.id = p.customer_id
    LEFT JOIN penjualan_detail pd ON p.id = pd.penjualan_id
    $where_clause AND p.id IS NOT NULL
";

$summary_result = mysqli_query($conn, $summary_query);
if (!$summary_result) {
    die("Summary query error: " . mysqli_error($conn));
}
$summary = mysqli_fetch_assoc($summary_result);

// Customer category analysis
$category_query = "
    SELECT 
        CASE 
            WHEN COUNT(DISTINCT p.id) >= 5 THEN 'VIP (5+ transaksi)'
            WHEN COUNT(DISTINCT p.id) >= 2 THEN 'Regular (2-4 transaksi)'
            ELSE 'New (1 transaksi)'
        END as kategori_customer,
        COUNT(DISTINCT c.id) as jumlah_customer,
        SUM(pd.qty * pd.harga) as total_penjualan,
        AVG(pd.qty * pd.harga) as rata_rata_penjualan,
        AVG(COUNT(DISTINCT p.id)) as rata_rata_frekuensi
    FROM customer c
    LEFT JOIN penjualan p ON c.id = p.customer_id
    LEFT JOIN penjualan_detail pd ON p.id = pd.penjualan_id
    $where_clause AND p.id IS NOT NULL
    GROUP BY c.id
    HAVING COUNT(DISTINCT p.id) > 0
";

// Create subquery and group by category - Fixed version
$category_full_query = "
    SELECT 
        kategori_customer,
        COUNT(*) as jumlah_customer,
        SUM(total_penjualan) as total_penjualan,
        AVG(total_penjualan) as rata_rata_penjualan,
        AVG(frekuensi) as rata_rata_frekuensi,
        CASE 
            WHEN SUM(total_penjualan) > 0 THEN 
                (SUM(total_penjualan) / (SELECT SUM(pd3.qty * pd3.harga) 
                                        FROM customer c3 
                                        LEFT JOIN penjualan p3 ON c3.id = p3.customer_id
                                        LEFT JOIN penjualan_detail pd3 ON p3.id = pd3.penjualan_id
                                        WHERE c3.aktif = 1 AND p3.id IS NOT NULL" . 
                                        ($periode != 'all' ? " AND DATE_FORMAT(p3.tanggal, '%Y-%m') = '$periode'" : '') . 
                                        (!empty($kota) ? " AND c3.kota = '" . mysqli_real_escape_string($conn, $kota) . "'" : '') . 
                                        ($customer_id > 0 ? " AND c3.id = $customer_id" : '') . ") * 100)
            ELSE 0 
        END as kontribusi_persen
    FROM (
        SELECT 
            c.id,
            CASE 
                WHEN COUNT(DISTINCT p.id) >= 5 THEN 'VIP (5+ transaksi)'
                WHEN COUNT(DISTINCT p.id) >= 2 THEN 'Regular (2-4 transaksi)'
                ELSE 'New (1 transaksi)'
            END as kategori_customer,
            SUM(pd.qty * pd.harga) as total_penjualan,
            COUNT(DISTINCT p.id) as frekuensi
        FROM customer c
        LEFT JOIN penjualan p ON c.id = p.customer_id
        LEFT JOIN penjualan_detail pd ON p.id = pd.penjualan_id
        WHERE c.aktif = 1 AND p.id IS NOT NULL" . 
        ($periode != 'all' ? " AND DATE_FORMAT(p.tanggal, '%Y-%m') = '$periode'" : '') . 
        (!empty($kota) ? " AND c.kota = '" . mysqli_real_escape_string($conn, $kota) . "'" : '') . 
        ($customer_id > 0 ? " AND c.id = $customer_id" : '') . "
        GROUP BY c.id
        HAVING frekuensi > 0
    ) as customer_stats
    GROUP BY kategori_customer
    ORDER BY AVG(total_penjualan) DESC
";

$category_result = mysqli_query($conn, $category_full_query);
if (!$category_result) {
    die("Category query error: " . mysqli_error($conn));
}

// Geographic analysis
$geo_query = "
    SELECT 
        c.kota,
        COUNT(DISTINCT c.id) as jumlah_customer,
        COUNT(DISTINCT p.id) as total_transaksi,
        SUM(pd.qty * pd.harga) as total_nominal,
        AVG(pd.qty * pd.harga) as rata_rata_transaksi,
        (SELECT c2.nama FROM customer c2 
         LEFT JOIN penjualan p2 ON c2.id = p2.customer_id
         LEFT JOIN penjualan_detail pd2 ON p2.id = pd2.penjualan_id
         WHERE c2.kota = c.kota AND c2.aktif = 1 AND p2.id IS NOT NULL
         GROUP BY c2.id 
         ORDER BY SUM(pd2.qty * pd2.harga) DESC 
         LIMIT 1) as customer_terbesar
    FROM customer c
    LEFT JOIN penjualan p ON c.id = p.customer_id
    LEFT JOIN penjualan_detail pd ON p.id = pd.penjualan_id
    $where_clause AND c.kota IS NOT NULL AND p.id IS NOT NULL
    GROUP BY c.kota
    ORDER BY total_nominal DESC
    LIMIT 10
";

$geo_result = mysqli_query($conn, $geo_query);
if (!$geo_result) {
    die("Geographic query error: " . mysqli_error($conn));
}

// Handle Excel Export
if ($export == 'excel') {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="laporan_customer_' . date('Y-m-d') . '.xls"');
    
    echo "<h3>LAPORAN RELASI CUSTOMER & TRANSAKSI</h3>";
    echo "<table border='1'>";
    echo "<tr>";
    echo "<th>Rank</th>";
    echo "<th>Customer Code</th>";
    echo "<th>Customer Name</th>";
    echo "<th>Kota</th>";
    echo "<th>Jenis</th>";
    echo "<th>Total Transaksi</th>";
    echo "<th>Nominal Total</th>";
    echo "<th>Rata-rata Transaksi</th>";
    echo "<th>Total Laba</th>";
    echo "<th>Margin %</th>";
    echo "<th>Transaksi Terakhir</th>";
    echo "</tr>";
    
    mysqli_data_seek($ranking_result, 0);
    $rank = 1;
    while ($row = mysqli_fetch_assoc($ranking_result)) {
        echo "<tr>";
        echo "<td>" . $rank . "</td>";
        echo "<td>" . htmlspecialchars($row['customer_kode']) . "</td>";
        echo "<td>" . htmlspecialchars($row['customer_nama']) . "</td>";
        echo "<td>" . htmlspecialchars($row['kota']) . "</td>";
        echo "<td>" . htmlspecialchars($row['jenis']) . "</td>";
        echo "<td>" . $row['total_transaksi'] . "</td>";
        echo "<td>" . $row['total_nominal'] . "</td>";
        echo "<td>" . $row['rata_rata_transaksi'] . "</td>";
        echo "<td>" . $row['total_laba'] . "</td>";
        echo "<td>" . number_format($row['margin_persen'], 1) . "%</td>";
        echo "<td>" . formatTanggal($row['transaksi_terakhir']) . "</td>";
        echo "</tr>";
        $rank++;
    }
    echo "</table>";
    exit;
}

// Get periode options dynamically
$periode_query = "SELECT DISTINCT DATE_FORMAT(tanggal, '%Y-%m') as value, DATE_FORMAT(tanggal, '%M %Y') as label FROM penjualan WHERE tanggal IS NOT NULL ORDER BY value DESC LIMIT 12";
$periode_result = mysqli_query($conn, $periode_query);

// Get customer options dynamically
$customer_options_query = "SELECT DISTINCT c.id, c.kode, c.nama FROM customer c INNER JOIN penjualan p ON c.id = p.customer_id WHERE c.aktif = 1 ORDER BY c.nama LIMIT 50";
$customer_options_result = mysqli_query($conn, $customer_options_query);

// Get kota options dynamically  
$kota_query = "SELECT DISTINCT c.kota FROM customer c INNER JOIN penjualan p ON c.id = p.customer_id WHERE c.aktif = 1 AND c.kota IS NOT NULL ORDER BY c.kota";
$kota_result = mysqli_query($conn, $kota_query);
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Laporan Relasi Customer & Transaksi - SBA System</title>
    
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
            font-size: 13px;
        }

        .container {
            max-width: 100%;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #333;
            padding-bottom: 15px;
        }

        .header h1 {
            margin: 0;
            color: #333;
            font-size: 24px;
        }

        .filter-section {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #ddd;
        }

        .filter-section h3 {
            margin: 0 0 15px 0;
            color: #333;
        }

        .form-group {
            display: inline-block;
            margin-right: 20px;
            margin-bottom: 10px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }

        .form-group select {
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 13px;
            min-width: 150px;
        }

        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 13px;
            text-decoration: none;
            display: inline-block;
            margin-right: 10px;
        }

        .btn-primary {
            background: #007bff;
            color: white;
        }

        .btn-success {
            background: #28a745;
            color: white;
        }

        .btn:hover {
            opacity: 0.8;
        }

        .summary-section {
            background: #e8f4fd;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #bee5eb;
        }

        .summary-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }

        .summary-item {
            text-align: center;
            background: white;
            padding: 15px;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }

        .summary-item h4 {
            margin: 0;
            color: #0c5460;
            font-size: 18px;
            font-weight: bold;
        }

        .summary-item p {
            margin: 5px 0 0 0;
            color: #666;
        }

        .table-container {
            overflow-x: auto;
            border: 1px solid #ddd;
            border-radius: 5px;
            margin-bottom: 30px;
        }

        .table {
            width: 100%;
            border-collapse: collapse;
            margin: 0;
            background: white;
        }

        .table th {
            background: #f8f9fa;
            color: #333;
            font-weight: bold;
            padding: 12px 8px;
            text-align: left;
            border-bottom: 2px solid #dee2e6;
            position: sticky;
            top: 0;
            font-size: 12px;
            white-space: nowrap;
        }

        .table td {
            padding: 10px 8px;
            border-bottom: 1px solid #dee2e6;
            vertical-align: top;
            font-size: 12px;
        }

        .table tbody tr:hover {
            background: #f8f9fa;
        }

        .table tbody tr:nth-child(even) {
            background: #fafafa;
        }

        .amount {
            text-align: right;
            font-weight: bold;
        }

        .profit {
            color: #28a745;
        }

        .text-center {
            text-align: center;
        }

        .ranking {
            background: #fff3cd;
            font-weight: bold;
            text-align: center;
            color: #856404;
        }

        .vip-customer {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            color: #856404;
            font-weight: bold;
        }

        .regular-customer {
            background: #ffffff;
            border: 2px solid #1976d2;
            color: #1976d2;
            font-weight: bold;
        }

        .new-customer {
            background: #e91e63;
            border: 1px solid #c2185b;
            color: white;
            font-weight: bold;
        }

        .section-header {
            background: #343a40;
            color: white;
            padding: 12px 15px;
            margin: 30px 0 15px 0;
            border-radius: 5px;
            font-weight: bold;
            font-size: 16px;
        }

        .tab-buttons {
            margin-bottom: 20px;
        }

        .tab-button {
            padding: 10px 20px;
            border: 1px solid #ddd;
            background: #f8f9fa;
            cursor: pointer;
            border-radius: 5px 5px 0 0;
            margin-right: 5px;
            display: inline-block;
        }

        .tab-button.active {
            background: #007bff;
            color: white;
            border-color: #007bff;
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        .chart-container {
            background: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }

        @media print {
            .filter-section, .btn, .tab-buttons {
                display: none;
            }
            
            body {
                background: white;
                font-size: 11px;
            }
            
            .table th, .table td {
                padding: 5px;
                font-size: 10px;
            }

            .tab-content {
                display: block !important;
            }
        }

        @media (max-width: 768px) {
            .form-group {
                display: block;
                margin-right: 0;
                margin-bottom: 15px;
            }
            
            .summary-grid {
                grid-template-columns: 1fr;
            }
            
            .table {
                font-size: 11px;
            }
            
            .table th, .table td {
                padding: 6px 4px;
            }
        }
    </style>
</head>
<body>

<div class="container">
    <div class="header">
        <h1>📊 LAPORAN RELASI CUSTOMER & TRANSAKSI</h1>
        <p>Sistem Manajemen Bisnis SBA</p>
    </div>

    <!-- Filter Section -->
    <div class="filter-section">
        <h3>🔍 Filter Laporan</h3>
        <form method="GET" action="">
            <div class="form-group">
                <label>Periode:</label>
                <select name="periode">
                    <option value="all" <?php echo $periode == 'all' ? 'selected' : ''; ?>>Semua Periode</option>
                    <?php 
                    if ($periode_result && mysqli_num_rows($periode_result) > 0) {
                        while ($periode_row = mysqli_fetch_assoc($periode_result)) {
                            $selected = $periode == $periode_row['value'] ? 'selected' : '';
                            echo "<option value='{$periode_row['value']}' $selected>{$periode_row['label']}</option>";
                        }
                    }
                    ?>
                </select>
            </div>
            
            <div class="form-group">
                <label>Customer:</label>
                <select name="customer_id">
                    <option value="0">Semua Customer</option>
                    <?php 
                    if ($customer_options_result && mysqli_num_rows($customer_options_result) > 0) {
                        while ($customer_row = mysqli_fetch_assoc($customer_options_result)) {
                            $selected = $customer_id == $customer_row['id'] ? 'selected' : '';
                            echo "<option value='{$customer_row['id']}' $selected>[{$customer_row['kode']}] {$customer_row['nama']}</option>";
                        }
                    }
                    ?>
                </select>
            </div>
            
            <div class="form-group">
                <label>Kota:</label>
                <select name="kota">
                    <option value="">Semua Kota</option>
                    <?php 
                    if ($kota_result && mysqli_num_rows($kota_result) > 0) {
                        while ($kota_row = mysqli_fetch_assoc($kota_result)) {
                            $selected = $kota == $kota_row['kota'] ? 'selected' : '';
                            echo "<option value='{$kota_row['kota']}' $selected>{$kota_row['kota']}</option>";
                        }
                    }
                    ?>
                </select>
            </div>
            
            <div class="form-group">
                <button type="submit" class="btn btn-primary">📋 Filter</button>
                <a href="?export=excel&<?php echo http_build_query($_GET); ?>" 
                   class="btn btn-success">📊 Export Excel</a>
                <button type="button" onclick="window.print()" class="btn btn-success">🖨️ Print</button>
            </div>
        </form>
    </div>

    <!-- Summary Section -->
    <div class="summary-section">
        <div class="summary-grid">
            <div class="summary-item">
                <h4><?php echo formatCurrency($summary['grand_total_penjualan'] ?? 0); ?></h4>
                <p>Total Transaksi</p>
            </div>
            <div class="summary-item">
                <h4><?php echo number_format($summary['total_customer_aktif'] ?? 0); ?></h4>
                <p>Customer Aktif</p>
            </div>
            <div class="summary-item">
                <h4><?php echo number_format($summary['total_transaksi'] ?? 0); ?></h4>
                <p>Total Transaksi</p>
            </div>
            <div class="summary-item">
                <h4 class="profit"><?php echo formatCurrency($summary['total_laba'] ?? 0); ?></h4>
                <p>Total Laba</p>
            </div>
            <div class="summary-item">
                <h4><?php echo number_format($summary['rata_rata_margin'] ?? 0, 1); ?>%</h4>
                <p>Margin Rata-rata</p>
            </div>
        </div>
    </div>

    <!-- Tab Buttons -->
    <div class="tab-buttons">
        <button class="tab-button active" onclick="showTab('ranking')">🏆 Ranking Customer</button>
        <button class="tab-button" onclick="showTab('analysis')">📈 Analisis Kategori</button>
        <button class="tab-button" onclick="showTab('geographic')">🌍 Analisis Geografis</button>
    </div>

    <!-- Ranking Customer Tab -->
    <div id="ranking" class="tab-content active">
        <div class="section-header">
            🏆 RANKING CUSTOMER BERDASARKAN NILAI TRANSAKSI
        </div>
        
        <div class="table-container">
            <table class="table">
                <thead>
                    <tr>
                        <th>Rank</th>
                        <th>Customer</th>
                        <th>Kota</th>
                        <th>Jenis</th>
                        <th>Total Transaksi</th>
                        <th>Nominal Total</th>
                        <th>Rata-rata/Transaksi</th>
                        <th>Total Laba</th>
                        <th>Margin %</th>
                        <th>Transaksi Terakhir</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    if ($ranking_result && mysqli_num_rows($ranking_result) > 0): 
                        $rank = 1;
                        while ($row = mysqli_fetch_assoc($ranking_result)): 
                            // Determine customer status
                            $status = 'New';
                            $status_class = 'new-customer';
                            if ($row['total_transaksi'] >= 5) {
                                $status = 'VIP';
                                $status_class = 'vip-customer';
                            } elseif ($row['total_transaksi'] >= 2) {
                                $status = 'Regular';
                                $status_class = 'regular-customer';
                            }
                            ?>
                            <tr>
                                <td class="ranking"><?php echo $rank; ?></td>
                                <td>
                                    <strong>[<?php echo htmlspecialchars($row['customer_kode']); ?>] <?php echo htmlspecialchars($row['customer_nama']); ?></strong><br>
                                    <small><?php echo htmlspecialchars($row['nama_toko'] ?? '-'); ?></small>
                                </td>
                                <td><?php echo htmlspecialchars($row['kota'] ?? '-'); ?></td>
                                <td><?php echo htmlspecialchars($row['jenis'] ?? '-'); ?></td>
                                <td class="text-center"><?php echo $row['total_transaksi']; ?></td>
                                <td class="amount profit"><?php echo formatCurrency($row['total_nominal']); ?></td>
                                <td class="amount"><?php echo formatCurrency($row['rata_rata_transaksi']); ?></td>
                                <td class="amount profit"><?php echo formatCurrency($row['total_laba']); ?></td>
                                <td class="text-center profit"><?php echo number_format($row['margin_persen'], 1); ?>%</td>
                                <td><?php echo formatTanggal($row['transaksi_terakhir']); ?></td>
                                <td><span class="<?php echo $status_class; ?>" style="padding: 3px 8px; border-radius: 12px; font-size: 11px;"><?php echo $status; ?></span></td>
                            </tr>
                            <?php 
                            $rank++;
                        endwhile; 
                    else: ?>
                        <tr>
                            <td colspan="11" class="text-center">Tidak ada data untuk filter yang dipilih</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Analysis Tab -->
    <div id="analysis" class="tab-content">
        <div class="section-header">
            📈 ANALISIS SEGMENTASI CUSTOMER
        </div>
        
        <div class="chart-container">
            <h4>Kategori Customer Berdasarkan Frekuensi Transaksi</h4>
            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Kategori Customer</th>
                            <th>Jumlah Customer</th>
                            <th>Total Penjualan</th>
                            <th>Rata-rata Penjualan</th>
                            <th>Rata-rata Frekuensi</th>
                            <th>Kontribusi %</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        if ($category_result && mysqli_num_rows($category_result) > 0): 
                            while ($category_row = mysqli_fetch_assoc($category_result)): 
                                $class = '';
                                if (strpos($category_row['kategori_customer'], 'VIP') !== false) {
                                    $class = 'vip-customer';
                                } elseif (strpos($category_row['kategori_customer'], 'Regular') !== false) {
                                    $class = 'regular-customer';
                                } else {
                                    $class = 'new-customer';
                                }
                                ?>
                                <tr class="<?php echo $class; ?>">
                                    <td><strong><?php echo htmlspecialchars($category_row['kategori_customer']); ?></strong></td>
                                    <td class="text-center"><?php echo $category_row['jumlah_customer']; ?></td>
                                    <td class="amount profit"><?php echo formatCurrency($category_row['total_penjualan']); ?></td>
                                    <td class="amount"><?php echo formatCurrency($category_row['rata_rata_penjualan']); ?></td>
                                    <td class="text-center"><?php echo number_format($category_row['rata_rata_frekuensi'], 1); ?></td>
                                    <td class="text-center profit"><?php echo number_format($category_row['kontribusi_persen'], 1); ?>%</td>
                                </tr>
                            <?php endwhile; 
                        else: ?>
                            <tr>
                                <td colspan="6" class="text-center">Tidak ada data kategori</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Geographic Analysis Tab -->
    <div id="geographic" class="tab-content">
        <div class="section-header">
            🌍 ANALISIS GEOGRAFIS CUSTOMER
        </div>
        
        <div class="table-container">
            <table class="table">
                <thead>
                    <tr>
                        <th>Kota</th>
                        <th>Jumlah Customer</th>
                        <th>Total Transaksi</th>
                        <th>Total Nominal</th>
                        <th>Rata-rata Transaksi</th>
                        <th>Customer Terbesar</th>
                        <th>Potensi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    if ($geo_result && mysqli_num_rows($geo_result) > 0): 
                        while ($geo_row = mysqli_fetch_assoc($geo_result)): 
                            // Determine potential based on total nominal
                            $potensi = 'Rendah';
                            $potensi_color = '#dc3545';
                            if ($geo_row['total_nominal'] > 200000000) {
                                $potensi = 'Tinggi';
                                $potensi_color = '#28a745';
                            } elseif ($geo_row['total_nominal'] > 50000000) {
                                $potensi = 'Sedang';
                                $potensi_color = '#ffc107';
                            }
                            ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($geo_row['kota']); ?></strong></td>
                                <td class="text-center"><?php echo $geo_row['jumlah_customer']; ?></td>
                                <td class="text-center"><?php echo $geo_row['total_transaksi']; ?></td>
                                <td class="amount profit"><?php echo formatCurrency($geo_row['total_nominal']); ?></td>
                                <td class="amount"><?php echo formatCurrency($geo_row['rata_rata_transaksi']); ?></td>
                                <td><?php echo htmlspecialchars($geo_row['customer_terbesar'] ?? '-'); ?></td>
                                <td>
                                    <span style="background: <?php echo $potensi_color; ?>; color: <?php echo $potensi == 'Sedang' ? '#212529' : 'white'; ?>; padding: 3px 8px; border-radius: 12px; font-size: 11px;">
                                        <?php echo $potensi; ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endwhile; 
                    else: ?>
                        <tr>
                            <td colspan="7" class="text-center">Tidak ada data geografis</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <div style="margin-top: 20px; text-align: center; color: #666; font-size: 12px;">
        <p>Laporan dibuat pada: <?php echo date('d F Y H:i:s'); ?></p>
        <p>Total Customer Aktif: <?php echo $summary['total_customer_aktif']; ?> | Total Record: <?php echo $summary['total_transaksi']; ?> transaksi</p>
    </div>
</div>

<script>
function showTab(tabName) {
    // Hide all tab contents
    var tabContents = document.querySelectorAll('.tab-content');
    for (var i = 0; i < tabContents.length; i++) {
        tabContents[i].classList.remove('active');
    }
    
    // Remove active class from all tab buttons
    var tabButtons = document.querySelectorAll('.tab-button');
    for (var i = 0; i < tabButtons.length; i++) {
        tabButtons[i].classList.remove('active');
    }
    
    // Show selected tab content
    document.getElementById(tabName).classList.add('active');
    
    // Add active class to clicked button
    event.target.classList.add('active');
}

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    showTab('ranking');
});
</script>

</body>
</html>