<?php
session_start();

// Debug session
error_log("Session check - Username: " . (isset($_SESSION['username']) ? $_SESSION['username'] : 'not set'));

// Include config without redirect if possible
include 'config.php';

// Check if user is logged in - but don't redirect if this is already a redirect
if (!isset($_SESSION['username'])) {
    error_log("User not logged in, redirecting to login.php");
    header("Location: login.php");
    exit();
}

// Get search parameters
$search = isset($_GET['search']) ? $_GET['search'] : '';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : '';
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : '';

// Build query with profit calculation
$query = "SELECT dp.*, c.nama as customer_nama, c.kota,
          COALESCE(SUM(dpd.jumlah * dpd.margin_rupiah), 0) as total_keuntungan,
          COALESCE(AVG(dpd.margin_persen), 0) as avg_margin_persen
          FROM draft_penjualan dp 
          LEFT JOIN customer c ON dp.customer_id = c.id 
          LEFT JOIN draft_penjualan_detail dpd ON dp.id = dpd.draft_id
          WHERE 1=1";

if (!empty($search)) {
    $search_escaped = mysqli_real_escape_string($conn, $search);
    $query .= " AND (c.nama LIKE '%$search_escaped%' OR dp.keterangan LIKE '%$search_escaped%')";
}

if (!empty($date_from)) {
    $date_from_escaped = mysqli_real_escape_string($conn, $date_from);
    $query .= " AND dp.tanggal >= '$date_from_escaped'";
}

if (!empty($date_to)) {
    $date_to_escaped = mysqli_real_escape_string($conn, $date_to);
    $query .= " AND dp.tanggal <= '$date_to_escaped'";
}

$query .= " GROUP BY dp.id ORDER BY dp.created_at DESC";

$result = mysqli_query($conn, $query);

if (!$result) {
    error_log("Query error: " . mysqli_error($conn));
    die("Database error occurred. Please check the logs.");
}

// Calculate summary totals
$total_amount_all = 0;
$total_keuntungan_all = 0;
$draft_count = 0;

// Store results in array for summary calculation
$draft_results = [];
while ($row = mysqli_fetch_assoc($result)) {
    $draft_results[] = $row;
    $total_amount_all += $row['total_amount'];
    $total_keuntungan_all += $row['total_keuntungan'];
    $draft_count++;
}

// Calculate overall margin percentage
$overall_margin_persen = ($total_amount_all > 0) ? (($total_keuntungan_all / ($total_amount_all - $total_keuntungan_all)) * 100) : 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Daftar Draft Penjualan - SBA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .table-responsive {
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .btn-group-sm .btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
        }
        .search-box {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            color: white;
        }
        .summary-box {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            color: white;
        }
        .profit-positive {
            color: #28a745;
            font-weight: bold;
        }
        .profit-negative {
            color: #dc3545;
            font-weight: bold;
        }
        .margin-high {
            background-color: #d4edda;
            color: #155724;
        }
        .margin-medium {
            background-color: #fff3cd;
            color: #856404;
        }
        .margin-low {
            background-color: #f8d7da;
            color: #721c24;
        }
    </style>
</head>
<body>
    <?php include 'navbar.php'; ?>
    
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-md-12">
                <!-- Alert Messages -->
                <?php if (isset($_GET['success'])): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    Draft penjualan berhasil dibuat!
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if (isset($_GET['updated'])): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    Draft berhasil diperbarui!
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if (isset($_GET['deleted'])): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    Draft berhasil dihapus!
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if (isset($_GET['error'])): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?php 
                    switch($_GET['error']) {
                        case '1': echo 'ID draft tidak ditemukan!'; break;
                        case '2': echo 'Draft tidak ditemukan di database!'; break;
                        case '3': echo 'Gagal menghapus draft!'; break;
                        default: echo 'Terjadi kesalahan!';
                    }
                    ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h2>Daftar Draft Penjualan</h2>
                    <a href="draft_penjualan_form.php" class="btn btn-primary">
                        Buat Draft Baru
                    </a>
                </div>

                <!-- Summary Box -->
                <?php if ($draft_count > 0): ?>
                <div class="summary-box">
                    <div class="row text-center">
                        <div class="col-md-3">
                            <h5>Total Draft</h5>
                            <h3><?php echo number_format($draft_count, 0, ',', '.'); ?></h3>
                        </div>
                        <div class="col-md-3">
                            <h5>Total Penjualan</h5>
                            <h3>Rp <?php echo number_format($total_amount_all, 0, ',', '.'); ?></h3>
                        </div>
                        <div class="col-md-3">
                            <h5>Total Estimasi Keuntungan</h5>
                            <h3>Rp <?php echo number_format($total_keuntungan_all, 0, ',', '.'); ?></h3>
                        </div>
                        <div class="col-md-3">
                            <h5>Margin Keseluruhan</h5>
                            <h3><?php echo number_format($overall_margin_persen, 1); ?>%</h3>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Search Form -->
                <div class="search-box">
                    <form method="GET" class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label">Cari Customer/Keterangan</label>
                            <input type="text" class="form-control" name="search" value="<?php echo $search; ?>" placeholder="Nama customer atau keterangan...">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Tanggal Dari</label>
                            <input type="date" class="form-control" name="date_from" value="<?php echo $date_from; ?>">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Tanggal Sampai</label>
                            <input type="date" class="form-control" name="date_to" value="<?php echo $date_to; ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button type="submit" class="btn btn-light">
                                    Cari
                                </button>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Results -->
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>No</th>
                                <th>Tanggal</th>
                                <th>Customer</th>
                                <th>Kota</th>
                                <th>Keterangan</th>
                                <th>Total Amount</th>
                                <th>Estimasi Keuntungan</th>
                                <th>Margin %</th>
                                <th>Dibuat</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $no = 1;
                            if (count($draft_results) > 0):
                                foreach ($draft_results as $row): 
                                    // Determine margin class for color coding
                                    $margin_class = '';
                                    if ($row['avg_margin_persen'] >= 10) {
                                        $margin_class = 'margin-high';
                                    } elseif ($row['avg_margin_persen'] >= 5) {
                                        $margin_class = 'margin-medium';
                                    } else {
                                        $margin_class = 'margin-low';
                                    }
                                    
                                    $profit_class = $row['total_keuntungan'] >= 0 ? 'profit-positive' : 'profit-negative';
                            ?>
                            <tr>
                                <td><?php echo $no++; ?></td>
                                <td><?php echo date('d/m/Y', strtotime($row['tanggal'])); ?></td>
                                <td><?php echo htmlspecialchars($row['customer_nama']); ?></td>
                                <td><?php echo htmlspecialchars($row['kota']); ?></td>
                                <td><?php echo htmlspecialchars(substr($row['keterangan'], 0, 30)); ?><?php echo strlen($row['keterangan']) > 30 ? '...' : ''; ?></td>
                                <td class="text-end">Rp <?php echo number_format($row['total_amount'], 0, ',', '.'); ?></td>
                                <td class="text-end <?php echo $profit_class; ?>">
                                    Rp <?php echo number_format($row['total_keuntungan'], 0, ',', '.'); ?>
                                </td>
                                <td class="text-center">
                                    <span class="badge <?php echo $margin_class; ?>">
                                        <?php echo number_format($row['avg_margin_persen'], 1); ?>%
                                    </span>
                                </td>
                                <td><?php echo date('d/m/Y H:i', strtotime($row['created_at'])); ?></td>
                                <td>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <a href="draft_penjualan_form.php?id=<?php echo $row['id']; ?>" class="btn btn-warning" title="Edit">
                                            Edit
                                        </a>
                                        <a href="javascript:void(0)" onclick="deleteDraft(<?php echo $row['id']; ?>)" class="btn btn-danger" title="Hapus">
                                            Hapus
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php 
                                endforeach;
                            else:
                            ?>
                            <tr>
                                <td colspan="10" class="text-center text-muted py-4">
                                    <p class="mb-0">Tidak ada draft penjualan ditemukan</p>
                                </td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <script>
    function deleteDraft(id) {
        Swal.fire({
            title: 'Konfirmasi Hapus',
            text: 'Yakin ingin menghapus draft penjualan ini?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Ya, Hapus!',
            cancelButtonText: 'Batal'
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = 'draft_penjualan_delete.php?id=' + id;
            }
        });
    }
    </script>
</body>
</html>